/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.validator.routines;

import java.io.BufferedReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.validator.routines.DateValidator;
import org.apache.commons.validator.routines.IBANValidator;
import org.apache.commons.validator.routines.IBANValidatorStatus;
import org.apache.commons.validator.routines.checkdigit.IBANCheckDigit;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.FieldSource;
import org.junit.jupiter.params.provider.MethodSource;

class IBANValidatorTest {
    private static final IBANValidator VALIDATOR = IBANValidator.getInstance();
    private static final String IBAN_PART = "(?:(\\d+)!([acn]))";
    private static final Pattern IBAN_PAT = Pattern.compile("(?:(\\d+)!([acn]))(?:(\\d+)!([acn]))(?:(\\d+)!([acn]))(?:(\\d+)!([acn]))?(?:(\\d+)!([acn]))?(?:(\\d+)!([acn]))?(?:(\\d+)!([acn]))?");
    private static final String IBAN_REGISTRY = "iban_registry_v99.txt";
    private static final Charset IBAN_REGISTRY_CHARSET = Charset.forName("windows-1252");
    private static final int MS_PER_DAY = 86400000;
    private static final long MAX_AGE_DAYS = 180L;
    private static final List<String> VALID_IBAN_FIXTURES = Arrays.asList("AD1200012030200359100100", "AE070331234567890123456", "AL47212110090000000235698741", "AT611904300234573201", "AZ21NABZ00000000137010001944", "BA391290079401028494", "BE68539007547034", "BG80BNBG96611020345678", "BH67BMAG00001299123456", "BI4210000100010000332045181", "BR1800000000141455123924100C2", "BR1800360305000010009795493C1", "BR9700360305000010009795493P1", "BY13NBRB3600900000002Z00AB00", "CH9300762011623852957", "CR05015202001026284066", "CY17002001280000001200527600", "CZ6508000000192000145399", "CZ9455000000001011038930", "DE89370400440532013000", "DJ2110002010010409943020008", "DK5000400440116243", "DO28BAGR00000001212453611324", "EE382200221020145685", "EG380019000500000000263180002", "ES9121000418450200051332", "FI2112345600000785", "FI5542345670000081", "AX2112345600000785", "AX5542345670000081", "FK88SC123456789012", "FO6264600001631634", "FR1420041010050500013M02606", "BL6820041010050500013M02606", "GF4120041010050500013M02606", "GP1120041010050500013M02606", "MF8420041010050500013M02606", "MQ5120041010050500013M02606", "NC8420041010050500013M02606", "PF5720041010050500013M02606", "PM3620041010050500013M02606", "RE4220041010050500013M02606", "TF2120041010050500013M02606", "WF9120041010050500013M02606", "YT3120041010050500013M02606", "GB29NWBK60161331926819", "GE29NB0000000101904917", "GI75NWBK000000007099453", "GL8964710001000206", "GR1601101250000000012300695", "GT82TRAJ01020000001210029690", "HN88CABF00000000000250005469", "HR1210010051863000160", "HU42117730161111101800000000", "IE29AIBK93115212345678", "IL620108000000099999999", "IQ98NBIQ850123456789012", "IS140159260076545510730339", "IT60X0542811101000000123456", "JO94CBJO0010000000000131000302", "KW81CBKU0000000000001234560101", "KZ86125KZT5004100100", "LB62099900000001001901229114", "LC55HEMM000100010012001200023015", "LI21088100002324013AA", "LT121000011101001000", "LU280019400644750000", "LY83002048000020100120361", "LV80BANK0000435195001", "LY83002048000020100120361", "MC5811222000010123456789030", "MD24AG000225100013104168", "ME25505000012345678951", "MK07250120000058984", "MN121234123456789123", "MR1300020001010000123456753", "MT84MALT011000012345MTLCAST001S", "MU17BOMM0101101030300200000MUR", "NI45BAPR00000013000003558124", "NL91ABNA0417164300", "NO9386011117947", "OM810180000001299123456", "PK36SCBL0000001123456702", "PL61109010140000071219812874", "PS92PALS000000000400123456702", "PT50000201231234567890154", "QA58DOHB00001234567890ABCDEFG", "RO49AAAA1B31007593840000", "RS35260005601001611379", "RU0204452560040702810412345678901", "SA0380000000608010167519", "SC18SSCB11010000000000001497USD", "SD8811123456789012", "SE4550000000058398257466", "SI56191000000123438", "SI56263300012039086", "SK3112000000198742637541", "SM86U0322509800000000270100", "SO211000001001000100141", "ST68000100010051845310112", "SV62CENR00000000000000700025", "SV43ACAT00000000000000123123", "TL380080012345678910157", "TN5910006035183598478831", "TR330006100519786457841326", "UA213223130000026007233566001", "UA213996220000026007233566001", "VA59001123000012345678", "VG96VPVG0000012345678901", "XK051212012345678906", "YE15CBYE0001018861234567891234");
    private static final List<String> INVALID_IBAN_FIXTURES = Arrays.asList("", "   ", "A", "AB", "FR1420041010050500013m02606", "MT84MALT011000012345mtlcast001s", "LI21088100002324013aa", "QA58DOHB00001234567890abcdefg", "RO49AAAA1b31007593840000", "LC62HEMM000100010012001200023015", "BY00NBRB3600000000000Z00AB00", "ST68000200010192194210112", "SV62CENR0000000000000700025", "NI04BAPR00000013000003558124", "RU1704452522540817810538091310419");

    IBANValidatorTest() {
    }

    private static String fmtRE(String ibanPat, int ibanLength) {
        int len;
        Matcher m = IBAN_PAT.matcher(ibanPat);
        if (!m.matches()) {
            throw new IllegalArgumentException("Unexpected IBAN pattern " + ibanPat);
        }
        StringBuilder sb = new StringBuilder();
        int totLen = len = Integer.parseInt(m.group(1));
        String curType = m.group(2);
        for (int i = 3; i <= m.groupCount() && m.group(i) != null; i += 2) {
            int count = Integer.parseInt(m.group(i));
            totLen += count;
            String type = m.group(i + 1);
            if (type.equals(curType)) {
                len += count;
                continue;
            }
            sb.append(IBANValidatorTest.formatToRE(curType, len));
            curType = type;
            len = count;
        }
        sb.append(IBANValidatorTest.formatToRE(curType, len));
        Assertions.assertEquals((int)ibanLength, (int)totLen, (String)("Wrong length for " + ibanPat));
        return sb.toString();
    }

    private static String formatToRE(String type, int len) {
        char ctype = type.charAt(0);
        switch (ctype) {
            case 'n': {
                return String.format("\\d{%d}", len);
            }
            case 'a': {
                return String.format("[A-Z]{%d}", len);
            }
            case 'c': {
                return String.format("[A-Z0-9]{%d}", len);
            }
        }
        throw new IllegalArgumentException("Unexpected type " + type);
    }

    static Collection<Arguments> ibanRegistrySource() throws Exception {
        Path ibanRegistry = Paths.get(IBANValidator.class.getResource(IBAN_REGISTRY).toURI());
        CSVFormat format = CSVFormat.DEFAULT.builder().setDelimiter('\t').build();
        BufferedReader rdr = Files.newBufferedReader(ibanRegistry, IBAN_REGISTRY_CHARSET);
        CSVRecord country = null;
        CSVRecord cc = null;
        CSVRecord additionalCc = null;
        CSVRecord structure = null;
        CSVRecord length = null;
        try (CSVParser p = new CSVParser((Reader)rdr, format);){
            for (CSVRecord o : p) {
                String item;
                switch (item = o.get(0)) {
                    case "Name of country": {
                        country = o;
                        break;
                    }
                    case "IBAN prefix country code (ISO 3166)": {
                        cc = o;
                        break;
                    }
                    case "Country code includes other countries/territories": {
                        additionalCc = o;
                        break;
                    }
                    case "IBAN structure": {
                        structure = o;
                        break;
                    }
                    case "IBAN length": {
                        length = o;
                        break;
                    }
                }
            }
        }
        Assertions.assertNotNull(country);
        Assertions.assertNotNull(cc);
        Assertions.assertNotNull(additionalCc);
        Assertions.assertNotNull(structure);
        Assertions.assertNotNull(length);
        ArrayList<Arguments> result = new ArrayList<Arguments>();
        for (int i = 1; i < country.size(); ++i) {
            String ac = additionalCc.get(i);
            List aCountry = Arrays.stream(ac.split(",")).filter(s -> !"N/A".equals(s)).map(s -> s.replace("(French part)", "")).map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toList());
            result.add(Arguments.of((Object[])new Object[]{country.get(i), cc.get(i), aCountry, Integer.parseInt(length.get(i)), structure.get(i)}));
        }
        return result;
    }

    static Collection<Arguments> ibanRegistrySourceExamples() throws Exception {
        Path ibanRegistry = Paths.get(IBANValidator.class.getResource(IBAN_REGISTRY).toURI());
        CSVFormat format = CSVFormat.DEFAULT.builder().setDelimiter('\t').build();
        BufferedReader rdr = Files.newBufferedReader(ibanRegistry, IBAN_REGISTRY_CHARSET);
        CSVRecord country = null;
        CSVRecord electronicExample = null;
        CSVRecord lastUpdateDate = null;
        try (CSVParser p = new CSVParser((Reader)rdr, format);){
            for (CSVRecord o : p) {
                String item;
                switch (item = o.get(0)) {
                    case "Name of country": {
                        country = o;
                        break;
                    }
                    case "IBAN electronic format example": {
                        electronicExample = o;
                        break;
                    }
                    case "Last update date": {
                        lastUpdateDate = o;
                        break;
                    }
                }
            }
        }
        Assertions.assertNotNull(country);
        int arraySize = country.size();
        Assertions.assertNotNull(electronicExample);
        Assertions.assertEquals((int)arraySize, (int)electronicExample.size());
        Assertions.assertNotNull(lastUpdateDate);
        Assertions.assertEquals((int)arraySize, (int)lastUpdateDate.size());
        ArrayList<Arguments> result = new ArrayList<Arguments>();
        Date lastDate = new Date(0L);
        String lastUpdated = null;
        for (int i = 1; i < country.size(); ++i) {
            result.add(Arguments.of((Object[])new Object[]{country.get(i), electronicExample.get(i)}));
            String mmyy = lastUpdateDate.get(i);
            Date dt = DateValidator.getInstance().validate(mmyy, "MMM-yy", Locale.ROOT);
            if (!dt.after(lastDate)) continue;
            lastDate = dt;
            lastUpdated = mmyy;
        }
        long age = (new Date().getTime() - lastDate.getTime()) / 86400000L;
        if (age > 180L) {
            System.out.println("WARNING: expected recent last update date, but found: " + lastUpdated);
        }
        return result;
    }

    public static Stream<Arguments> validateIbanStatuses() {
        return Stream.of(Arguments.of((Object[])new Object[]{"XX", IBANValidatorStatus.UNKNOWN_COUNTRY}), Arguments.of((Object[])new Object[]{"AD0101", IBANValidatorStatus.INVALID_LENGTH}), Arguments.of((Object[])new Object[]{"AD12XX012030200359100100", IBANValidatorStatus.INVALID_PATTERN}), Arguments.of((Object[])new Object[]{"AD9900012030200359100100", IBANValidatorStatus.INVALID_CHECKSUM}), Arguments.of((Object[])new Object[]{"AD1200012030200359100100", IBANValidatorStatus.VALID}));
    }

    @ParameterizedTest
    @MethodSource(value={"ibanRegistrySourceExamples"})
    void testExampleAccountsShouldBeValid(String countryName, String example) {
        Assumptions.assumeFalse((boolean)INVALID_IBAN_FIXTURES.contains(example), (String)("Skip invalid example: " + example + " for " + countryName));
        Assertions.assertTrue((boolean)IBANValidator.getInstance().isValid(example), (String)("IBAN validator returned false for " + example + " for " + countryName));
    }

    @Test
    void testGetRegexValidatorPatterns() {
        Assertions.assertNotNull((Object)VALIDATOR.getValidator("GB").getRegexValidator().getPatterns(), (String)"GB");
    }

    @Test
    void testGetValidator() {
        Assertions.assertNotNull((Object)VALIDATOR.getValidator("GB"), (String)"GB");
        Assertions.assertNull((Object)VALIDATOR.getValidator("gb"), (String)"gb");
    }

    @Test
    void testHasValidator() {
        Assertions.assertTrue((boolean)VALIDATOR.hasValidator("GB"), (String)"GB");
        Assertions.assertFalse((boolean)VALIDATOR.hasValidator("gb"), (String)"gb");
    }

    @ParameterizedTest
    @FieldSource(value={"INVALID_IBAN_FIXTURES"})
    void testInValid(String invalidIban) {
        Assertions.assertNotNull(INVALID_IBAN_FIXTURES);
        Assertions.assertFalse((boolean)VALIDATOR.isValid(invalidIban), (String)invalidIban);
    }

    @ParameterizedTest
    @FieldSource(value={"VALID_IBAN_FIXTURES"})
    void testMoreValid(String invalidIban) {
        Assertions.assertNotNull(VALID_IBAN_FIXTURES);
        Assertions.assertTrue((boolean)VALIDATOR.isValid(invalidIban), (String)invalidIban);
    }

    @Test
    void testNull() {
        Assertions.assertFalse((boolean)VALIDATOR.isValid(null), (String)"isValid(null)");
    }

    @Test
    void testSetDefaultValidator1() {
        IllegalStateException thrown = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> VALIDATOR.setValidator("GB", 15, "GB"));
        Assertions.assertEquals((Object)"The singleton validator cannot be modified", (Object)thrown.getMessage());
    }

    @Test
    void testSetDefaultValidator2() {
        IllegalStateException thrown = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> VALIDATOR.setValidator("GB", -1, "GB"));
        Assertions.assertEquals((Object)"The singleton validator cannot be modified", (Object)thrown.getMessage());
    }

    @Test
    void testSetValidatorLC() {
        IBANValidator validator = new IBANValidator();
        IllegalArgumentException thrown = (IllegalArgumentException)Assertions.assertThrows(IllegalArgumentException.class, () -> validator.setValidator("gb", 15, "GB"));
        Assertions.assertEquals((Object)"Invalid country Code; must be exactly 2 upper-case characters", (Object)thrown.getMessage());
    }

    @Test
    void testSetValidatorLen1() {
        IBANValidator validator = new IBANValidator();
        Assertions.assertNotNull((Object)validator.setValidator("GB", -1, ""), (String)"should be present");
        Assertions.assertNull((Object)validator.setValidator("GB", -1, ""), (String)"no longer present");
    }

    @Test
    void testSetValidatorLen35() {
        IBANValidator validator = new IBANValidator();
        IllegalArgumentException thrown = (IllegalArgumentException)Assertions.assertThrows(IllegalArgumentException.class, () -> validator.setValidator("GB", 35, "GB"));
        Assertions.assertEquals((Object)"Invalid length parameter, must be in range 8 to 34 inclusive: 35", (Object)thrown.getMessage());
    }

    @Test
    void testSetValidatorLen7() {
        IBANValidator validator = new IBANValidator();
        IllegalArgumentException thrown = (IllegalArgumentException)Assertions.assertThrows(IllegalArgumentException.class, () -> validator.setValidator("GB", 7, "GB"));
        Assertions.assertEquals((Object)"Invalid length parameter, must be in range 8 to 34 inclusive: 7", (Object)thrown.getMessage());
    }

    @Test
    void testSorted() {
        IBANValidator validator = new IBANValidator();
        IBANValidator.Validator[] vals = validator.getDefaultValidators();
        Assertions.assertNotNull((Object)vals);
        for (int i = 1; i < vals.length; ++i) {
            if (vals[i].countryCode.compareTo(vals[i - 1].countryCode) > 0) continue;
            Assertions.fail((String)("Not sorted: " + vals[i].countryCode + " <= " + vals[i - 1].countryCode));
        }
    }

    @ParameterizedTest
    @FieldSource(value={"VALID_IBAN_FIXTURES"})
    void testValid(String iban) {
        Assertions.assertTrue((boolean)IBANCheckDigit.IBAN_CHECK_DIGIT.isValid(iban), (String)("Checksum fail: " + iban));
        Assertions.assertTrue((boolean)VALIDATOR.hasValidator(iban), (String)("Missing validator: " + iban));
        Assertions.assertTrue((boolean)VALIDATOR.isValid(iban), (String)iban);
    }

    @ParameterizedTest
    @MethodSource(value={"validateIbanStatuses"})
    void testValidateIbanStatuses(String iban, IBANValidatorStatus expectedStatus) {
        Assertions.assertEquals((Object)expectedStatus, (Object)IBANValidator.getInstance().validate(iban));
    }

    @ParameterizedTest
    @MethodSource(value={"ibanRegistrySource"})
    void testValidatorShouldExistWithProperConfiguration(String countryName, String countryCode, List<String> acountyCode, int ibanLength, String structure) throws Exception {
        String countryInfo = " countryCode: " + countryCode + ", countryName: " + countryName;
        IBANValidator.Validator validator = IBANValidator.getInstance().getValidator(countryCode);
        Assertions.assertNotNull((Object)validator, (String)("IBAN validator returned null for" + countryInfo));
        Assertions.assertEquals((int)ibanLength, (int)validator.getIbanLength(), (String)("IBAN length should be " + ibanLength + " for" + countryInfo));
        List allPatterns = Arrays.stream(validator.getRegexValidator().getPatterns()).map(Pattern::pattern).collect(Collectors.toList());
        String re = IBANValidatorTest.fmtRE(structure.substring(2), ibanLength - 2);
        Assertions.assertTrue((boolean)allPatterns.remove(countryCode + re), (String)("No pattern " + countryCode + re + " found for " + countryInfo));
        for (String ac : acountyCode) {
            Assertions.assertTrue((boolean)allPatterns.remove(ac + re), (String)("No additional country code " + ac + " found for " + countryInfo));
        }
        Assertions.assertTrue((boolean)allPatterns.isEmpty(), (String)("Unrecognized patterns: " + allPatterns + " for" + countryInfo));
    }
}

